/*
 Copyright (C) 2002 Andreas Thiede ( a.thiede@berlin.de )

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#import <skins/DefaultSkin.hpp>
#import <IO/Output/Video/GLWindow.h>
#import <IO/Manager/IOManager.hpp>
#import <lib/Preferences.h>
#import <util/str.h>
#import <util/Loader.h>
#import <unistd.h>

#define NONE   0
#define TV     1
#define PLAYER 2

@interface MyButton : NSView {

@private

    NSString * mName;
    NSImage  * mImage;
    NSImage  * mAlternateImage;
    bool       mMouseDown;
}

- (void) setImage:(NSImage*) image;
- (void) setAlternateImage:(NSImage*) image;
- (void) initWithFrameAndName:(NSRect) rect name:(const char*) name;
- (void) drawRect:(NSRect)rect;
- (void) keyDown:(NSEvent *)theEvent;
- (void) mouseDown:(NSEvent *)theEvent;
- (void) mouseMoved:(NSEvent *)theEvent;
- (BOOL)acceptsFirstMouse:(NSEvent *)theEvent;
@end

@implementation MyButton

- (BOOL)acceptsFirstMouse:(NSEvent *)theEvent
{
    return YES;
}

- (void) initWithFrameAndName:(NSRect) rect name:(const char*) name
{
    [ super initWithFrame:rect ];
    
    mName = [ NSString stringWithCString:name ];
    
    [ mName retain ];
    
    mMouseDown = false;
    
    mImage = 0;
    mAlternateImage = 0;
}

- (void) setAlternateImage:(NSImage*) image
{
    if( mAlternateImage ) {
        [ mAlternateImage release ];
    }

    mAlternateImage = image;

    if( mAlternateImage ) {
        [ mAlternateImage retain ];
    }
}

- (void) setImage:(NSImage*) image
{
    if( mImage ) {
        [ mImage release ];
    }

    mImage = image;

    if( mImage ) {
        [ mImage retain ];
    }
}

-(void)drawRect:(NSRect)rect
{
    //puts("MyButton.drawRect");
    
    [ super drawRect:rect ];
    
    [ self lockFocus ];
    
    [[NSColor clearColor] set];
    
    NSRectFill( [ self frame ] );
    
    float srcW = [ mImage size].width;
    float srcH = [ mImage size].height;
    
    if( mMouseDown && mAlternateImage ) {        
        [ mAlternateImage drawAtPoint:NSZeroPoint fromRect:NSMakeRect( 0, 0, srcW,srcH ) operation:NSCompositeSourceOver fraction:1.0 ];    
    }
    else if( mImage ) {
        [ mImage drawAtPoint:NSZeroPoint fromRect:NSMakeRect( 0, 0, srcW,srcH ) operation:NSCompositeSourceOver fraction:1.0 ];    
    }
    
    [ self unlockFocus ];
}

- (void)mouseDown:(NSEvent *) event
{
    //printf("MyButton.mouseDown in %s\n", [ mName cString ]);

    mMouseDown = true;

    [ self display ];
    
    //[super mouseDown:event];
    
    //[ self highlight: true ];
    
    //[ self display ];    
    
    //[ [ self window ] flushWindow ];
    
    if( strstr( [ mName cString ], "drag" ) != 0 ) {
        IOManager::instance().sendCommand( "WindowController", "buttonPressed", [ mName cString ] );
    }
#if 0
    else {

        bool keepOn;
    
        do {

            NSEvent * theEvent = [ [ self window ] nextEventMatchingMask: NSAnyEventMask ];
            
            switch( [theEvent type] ) {

            case NSLeftMouseUp:
            case NSRightMouseUp:
            case NSOtherMouseUp:
                keepOn = NO;
                break;
            case NSMouseExited:
                //puts("NSMouseExited");
                break;
            case NSMouseEntered:
                //puts("NSMouseEntered");
                break;
            default:
                /* Ignore any other kind of event. */
                break;
            }

            NSPoint mouseLoc = [self convertPoint:[theEvent locationInWindow] fromView:nil];

            //printf("mouse(%d,%d)\n", (int) mouseLoc.x, (int) mouseLoc.y );
        
            bool isInside = [self mouse:mouseLoc inRect:[self bounds]];
        
            if( isInside) 
                //printf( "mouse inside\n" );
            else
                //printf( "mouse outside\n" );
            
        } while (keepOn);            
    }
#endif
}

- (void)mouseUp:(NSEvent *) event
{
    mMouseDown = false;

    //printf("MyButton.mouseUp in %s\n", [ mName cString ]);

    [ self display ];
    
    //[super mouseUp:event];

    //[ self highlight: false ];
    
    //[ self display ];    

    [ [ self window ] flushWindow ];
     
    //[ self setState: NSOffState ];

    NSPoint mouseLoc = [ self convertPoint:[ event locationInWindow ] fromView:nil ];

    //printf("mouse(%d,%d)\n", (int) mouseLoc.x, (int) mouseLoc.y );
        
    bool isInside = [self mouse:mouseLoc inRect:[self bounds]];

#if 0    
    if( isInside) 
        //printf( "mouse inside\n" );
    else
        //printf( "mouse outside\n" );
#endif                
    if( isInside ) {
        IOManager::instance().sendCommand( "WindowController", "buttonPressed", [ mName cString ] );
    }
}

- (void)mouseMoved:(NSEvent *) event
{
    //puts("MyButton.mouseMoved");
    
    [super mouseMoved:event];
}

- (void)keyDown:(NSEvent *)theEvent
{
    //puts("MyButton.keyDown");
    
    [super keyDown:theEvent];
}

@end

@interface ControlWindow : NSWindow
{
}

@end // interface ControlWindow

static NSImage  *  defaultSkin      = 0;
static NSImage  *  forwardImage     = 0;
static NSImage  *  fastForwardImage = 0;
static NSImage  *  reverseImage     = 0;
static NSImage  *  fastReverseImage = 0;
static NSImage  *  exitImage        = 0;
static NSImage  *  stopImage        = 0;
static NSImage  *  pauseImage       = 0;
static NSImage  *  zoomImage        = 0;
static NSImage  *  playListImage    = 0;
static NSImage  *  movieImage       = 0;

static NSImage  *  pushedForwardImage     = 0;
static NSImage  *  pushedFastForwardImage = 0;
static NSImage  *  pushedReverseImage     = 0;
static NSImage  *  pushedFastReverseImage = 0;
static NSImage  *  pushedExitImage        = 0;
static NSImage  *  pushedStopImage        = 0;
static NSImage  *  pushedPauseImage       = 0;
static NSImage  *  pushedZoomImage        = 0;

static NSImage  *  upImage                = 0;
static NSImage  *  pushedUpImage          = 0;
static NSImage  *  downImage              = 0;
static NSImage  *  pushedDownImage        = 0;
static NSImage  *  recordImage            = 0;
static NSImage  *  pushedRecordImage      = 0;
static NSImage  *  beginImage             = 0;
static NSImage  *  pushedBeginImage       = 0;
static NSImage  *  endImage               = 0;
static NSImage  *  pushedEndImage         = 0;
static NSImage  *  pushedPlayListImage    = 0;
static NSImage  *  pushedMovieImage       = 0;


static NSImage  *  dragImage        = 0;
static float       imageWidth       = 0;
static float       imageHeight      = 0;

DefaultSkin::DefaultSkin()
 : SkinBase( "DefaultSkin" ), mWindow(0)
{
}
    
DefaultSkin::~DefaultSkin()
{
}

void DefaultSkin::update()
{
    //puts("DefaultSkin::update");
    
    if( ! [ mWindow isFullScreen ] ) {
    
        [ [ mWindow contentView ] lockFocus ];
    
        [[NSColor clearColor] set];
    
        NSRectFill( [ [ mWindow contentView ] frame ] );
    
        if( defaultSkin != 0 ) {
    
            float srcW = [ defaultSkin size].width;
            float srcH = [ defaultSkin size].height;
    
            float dstW = [[ mWindow contentView ] frame ].size.width;
            float dstH = [[ mWindow contentView ] frame ].size.height;

            //printf("DefaultSkin::update wh(%d,%d)\n", (int)dstW, (int)dstH );
        
            if( dstW > imageWidth || dstH > imageHeight ) {
                [ defaultSkin setSize: NSMakeSize( dstW, dstH ) ];
            }
        
            [ defaultSkin drawAtPoint:NSZeroPoint fromRect:NSMakeRect( 0, 0, srcW,srcH ) operation:NSCompositeSourceOver fraction:1.0 ];
    
        }

        if( mDrag != 0 ) {
    
            NSRect frame = [ mDrag frame ];

            frame.origin.x = [ mWindow frame ].size.width - [ dragImage size].width;
        
            [ mDrag setFrame:frame ];
        }

        [ [ mWindow contentView ] unlockFocus ];
    }
} 

NSImage * DefaultSkin::loadImage( const string & name )
{
    string path = [[ [ NSBundle mainBundle ] builtInPlugInsPath ] cString ];
    
    // printf("DefaultSkin.loadImage pluginPath is %s\n", path.c_str() );
    
    path += "/DefaultSkin.bundle/Contents/Resources/";
    path += name;
    path += ".tiff";

    NSImage * img = [ [ NSImage alloc ] initWithContentsOfFile:[ NSString stringWithCString:path.c_str() ] ];
    
    if( img ) {
        [ img retain ];
    }
    
    return img;
}

bool DefaultSkin::createSkinForWindow( GLWindow * win ) 
{
    if( win == 0 ) {
        return false;
    }

    mShowControls = true;
    mMode         = NONE;
    
    defaultSkin      = loadImage( "DefaultSkin" );
    forwardImage     = loadImage( "Forward"     );
    fastForwardImage = loadImage( "FastForward" );
    reverseImage     = loadImage( "Reverse"     );
    fastReverseImage = loadImage( "FastReverse" );
    exitImage        = loadImage( "Exit"        );
    stopImage        = loadImage( "Stop"        );
    pauseImage       = loadImage( "Pause"       );
    upImage          = loadImage( "Up"          );
    downImage        = loadImage( "Down"        );
    beginImage       = loadImage( "Begin"       );
    endImage         = loadImage( "End"         );
    recordImage      = loadImage( "Record"      );
    playListImage    = loadImage( "PlayList"    );
    movieImage       = loadImage( "Movie"       );
    zoomImage        = loadImage( "Zoom"        );

    pushedForwardImage     = loadImage( "PushedForward"     );
    pushedFastForwardImage = loadImage( "PushedFastForward" );
    pushedReverseImage     = loadImage( "PushedReverse"     );
    pushedFastReverseImage = loadImage( "PushedFastReverse" );
    pushedExitImage        = loadImage( "PushedExit"        );
    pushedStopImage        = loadImage( "PushedStop"        );
    pushedPauseImage       = loadImage( "PushedPause"       );

    pushedUpImage          = loadImage( "PushedUp"          );
    pushedDownImage        = loadImage( "PushedDown"        );
    pushedBeginImage       = loadImage( "PushedBegin"       );
    pushedEndImage         = loadImage( "PushedEnd"         );
    pushedRecordImage      = loadImage( "PushedRecord"      );
    pushedPlayListImage    = loadImage( "PushedPlayList"    );
    pushedMovieImage       = loadImage( "PushedMovie"       );
    pushedZoomImage        = loadImage( "PushedZoom"        );

    dragImage        = loadImage( "Drag"        );
    
    //printf("Image is %p\n", defaultSkin );
    
    if( defaultSkin != 0 ) {
    
        [ defaultSkin setScalesWhenResized:YES ];

        imageWidth  = [ defaultSkin size].width;
        imageHeight = [ defaultSkin size].height;
    }
    
    mWindow = win;
    
    [ mWindow setOpaque: NO ];
    [ mWindow setAlphaValue: 1.0 ];
    [ mWindow  setBackgroundColor: [NSColor clearColor]];

    [ mWindow setTopBorder:     12 ];
    [ mWindow setLeftBorder:    12 ];
    [ mWindow setRightBorder:   12 ];
    [ mWindow setBottomBorder:  60 ];

    createControls();

    NSSize screenBounds = [[ NSScreen mainScreen ] frame ].size;
    
    NSRect windowPos = NSMakeRect( 0,0, screenBounds.width, 60 );
    
    mControlWindow = [ NSWindow alloc ];
    
    int flags = NSBorderlessWindowMask;
    
    NSRect frame = [ NSWindow contentRectForFrameRect: windowPos styleMask: flags ];
    
    [ mControlWindow initWithContentRect: frame styleMask: flags backing: NSBackingStoreBuffered defer: NO ];

    [ mControlWindow setLevel: kCGOverlayWindowLevel ];    

    [ mControlWindow setOpaque: NO ];
    [ mControlWindow setAlphaValue: 1.0 ];
    [ mControlWindow setBackgroundColor: [NSColor clearColor]];
    
    update();
    
    return true;
}

MyButton * DefaultSkin::createButton( const char * aName, NSRect aRect, NSImage * image, NSImage * alternateImage )
{
    MyButton * button = [ MyButton alloc ];

    if( button ) {
        
        [ button initWithFrameAndName:aRect name:aName ];        
        
        if( image ) {
            [ button setImage:image ];
        }

        if( alternateImage ) {
            [ button setAlternateImage:alternateImage ];
        }        
        
        [ button retain ];
    }
    
    return button;
}

void DefaultSkin::createControls()
{
    int x = 0;
    
    mExit         = createButton( "exit",         NSMakeRect( x +  10, 12, 32, 32 ), exitImage,        pushedExitImage        );
    mBegin        = createButton( "begin",        NSMakeRect( x +  50, 12, 32, 32 ), beginImage, pushedBeginImage             );
    mFastBackward = createButton( "fastbackward", NSMakeRect( x +  90, 12, 32, 32 ), fastReverseImage, pushedFastReverseImage );
    mBackward     = createButton( "backward",     NSMakeRect( x + 130, 12, 32, 32 ), reverseImage,     pushedReverseImage     );
    mStop         = createButton( "stop",         NSMakeRect( x + 170, 12, 32, 32 ), stopImage,        pushedStopImage        );
    mPause1       = createButton( "pause",        NSMakeRect( x + 210, 12, 32, 32 ), pauseImage,       pushedPauseImage       );
    mForward      = createButton( "forward",      NSMakeRect( x + 250, 12, 32, 32 ), forwardImage,     pushedForwardImage     );
    mFastForward  = createButton( "fastforward",  NSMakeRect( x + 290, 12, 32, 32 ), fastForwardImage, pushedFastForwardImage );
    mEnd          = createButton( "end",          NSMakeRect( x + 330, 12, 32, 32 ), endImage, pushedEndImage                 );
    mPlayList1    = createButton( "playlist",     NSMakeRect( x + 370, 12, 32, 32 ), playListImage,    pushedPlayListImage    );
    mZoom1        = createButton( "zoom",         NSMakeRect( x + 410, 12, 32, 32 ), zoomImage,        pushedZoomImage        );

    mUp           = createButton( "up",           NSMakeRect( x +  50, 12, 32, 32 ), upImage, pushedUpImage );
    mDown         = createButton( "down",         NSMakeRect( x +  90, 12, 32, 32 ), downImage, pushedDownImage               );
    mPause2       = createButton( "pause",        NSMakeRect( x + 130, 12, 32, 32 ), pauseImage,       pushedPauseImage       );
    mRecord       = createButton( "record",       NSMakeRect( x + 170, 12, 32, 32 ), recordImage, pushedRecordImage           );
    mPlayList2    = createButton( "playlist",     NSMakeRect( x + 210, 12, 32, 32 ), playListImage,    pushedPlayListImage    );
    mMovie        = createButton( "playmovies",   NSMakeRect( x + 250, 12, 32, 32 ), movieImage,       pushedMovieImage       );
    mZoom2        = createButton( "zoom",         NSMakeRect( x + 290, 12, 32, 32 ), zoomImage,        pushedZoomImage        );


    //mDrag         = createButton( "lbdrag",       NSMakeRect( [ mWindow frame ].size.width - 20, 0, 20, 20 ), dragImage, 0    );
    mDrag = 0;
}

bool DefaultSkin::closeSkinForWindow( GLWindow * win )
{
    return true;
}

void DefaultSkin::showControls( bool on )
{
    mShowControls = on;

    updateWindowSize();
    
    updateControls();
}

void DefaultSkin::toggleFullScreen( bool isFullScreen )
{
    if( ! isFullScreen ) {
        [ mControlWindow orderOut:nil ];
    }
    else {
        [ mControlWindow orderWindow:NSWindowAbove relativeTo:[ mWindow windowNumber ] ];
    }
    
    updateControls();
}
    
void DefaultSkin::showTVControls()
{
    mMode = TV;

    updateControls();
}

void DefaultSkin::showPlayerControls()
{
    mMode = PLAYER;

    updateControls();
}

void DefaultSkin::updateWindowSize()
{
    if( [ mWindow isFullScreen ]) {
    
        if( mShowControls == false ) {
            [ mControlWindow orderOut:nil ];
        }
        else {
            [ mControlWindow orderWindow:NSWindowAbove relativeTo:[ mWindow windowNumber ] ];
        }
    }
    else {

        float x = [ mWindow frame ].origin.x;
        float y = [ mWindow frame ].origin.y;
        float w = [ mWindow frame ].size.width;
        float h = [ mWindow frame ].size.height;
        
        if( mShowControls ) {

            [ mWindow setTopBorder:     12 ];
            [ mWindow setLeftBorder:    12 ];
            [ mWindow setRightBorder:   12 ];
            [ mWindow setBottomBorder:  56 ];
            
            h += 72;
            y -= 72;
        }
        else {
            [ mWindow setTopBorder:     0 ];
            [ mWindow setLeftBorder:    0 ];
            [ mWindow setRightBorder:   0 ];
            [ mWindow setBottomBorder:  0 ];
            
            h -= 72;
            y += 72;
        }
        
        [ mWindow move:x y:y ];
        [ mWindow resize:w h:h ];
    }
}

void DefaultSkin::updateControls()
{
    NSWindow * win;
    
    if( ! [ mWindow isFullScreen ] ) {
        win = mWindow;
    }
    else {
        win = mControlWindow;
    }

    removeControls( mWindow );
    removeControls( mControlWindow );

    [ win display ];
    
    if( mShowControls ) {
    
        if( mMode == TV ) {

            //puts("showTVControls");
    
            showButton( mExit,      win );
            showButton( mUp,        win );
            showButton( mDown,      win );
            showButton( mRecord,    win );
            showButton( mPause2,    win );
            showButton( mPlayList2, win );
            showButton( mMovie,     win );
            showButton( mZoom2,     win );

        }
        else if( mMode == PLAYER ) {

            //puts("showPlayerControls");

            showButton( mExit,         win );
            showButton( mBegin,        win );
            showButton( mFastBackward, win );
            showButton( mBackward,     win );
            showButton( mForward,      win );
            showButton( mStop,         win );
            showButton( mPause1,       win );
            showButton( mFastForward,  win );
            showButton( mEnd,          win );
            showButton( mPlayList1,    win );
            showButton( mZoom1,        win );
        }

        if( ! [ mWindow isFullScreen ] ) {
            showButton( mDrag, win );
        }

        [ win display ];
    }
}

void DefaultSkin::showButton( MyButton * button, NSWindow * win )
{
    [ [ win contentView ] addSubview: button positioned:NSWindowAbove relativeTo:nil ];
}

void DefaultSkin::removeControls( NSWindow * win )
{
    NSArray * subViews = [[ win contentView ] subviews ];
        
    if( subViews ) {

 again:
         
        for( unsigned i = 0; i < [ subViews count ]; i++ ) {
            
            NSView * v = (NSView*) [ subViews objectAtIndex:i ];
                
            if( v != 0  && [ v isKindOfClass: [ MyButton class ]] ) {
                
                //printf("removing %s\n", [ ((MyButton*)v)->mName cString ] );
                    
                [ v removeFromSuperview ];
                goto again;
            }
        }
    }
}

NSImage * DefaultSkin::getWindowBackgroundImage()
{
    return loadImage( "DefaultSkin" );
}
    
bool DefaultSkin::command( const string & command, const string & param1,  const string & param2, const string & param3 )
{
    //printf("%s.command: %s %s %s\n", getName().c_str(), command.c_str(), param1.c_str(), param2.c_str() );

    return true;
}

extern "C" int initDefaultSkin( int argc, const char ** argv )
{
    //puts("DefaultSkin loaded");

    DefaultSkin * s = new DefaultSkin();
    
    IOManager::instance().registerObject( s );
    
    return true;
}
